<#
:: ========================================================================
:: Synchronize files to cloud/network storage for multiple PCs
::
:: Created (bat): 2021-11-09
:: Converted (ps1): 2024-12-20
:: Updated: 2025-11-19
::   - UI Templates use Pattern Mirror (Source of Truth Master). Character template file name: UI*template*.ini. Use naming compatible with EQ's copy UI.
::   - Moved user config closer to top for easier access.
::
:: 2025-01-07
::   - Mirror uifiles templates on MASTER (source of Truth). simple/sane approach for multiple PCs/installs
::   - add more folder configs to be more flexible (ie keep Emul isolated)
::
::
:: *********************
:: ***** IMPORTANT *****
:: *********************

    Configure (User configuration)
    * $Config
    * $Installs

	Mirrors
    Setup the Master ("MasterBase\$UIF Version\uifiles\<customUI>" folders) first to have appropriate directories for the MIRROR!!!!!

	Currently this is for custom uifiles. This is to make syncing changes
	easier with multiple installs/computers by having the Master be the source of truth.
    And make sure you have these for each new EQ install based on Version settings!!!!!

    Pattern Mirror
    Master ("MasterBase\$UIT Version\User Templates") is the source of truth for EQ Character UI INIs.
    Character template file name: UI*template*.ini. All copied down from master to installs.
    Deletes any matching the pattern from installs that do not exist on master.

:: *********************
:: ***** IMPORTANT *****
:: *********************
::
:: ========================================================================
#>

################################################################################
################################################################################

# definitions
$title = "Sync_Multi_EQ-ps"
$logFile = Join-Path -Path (Split-Path -Parent $PSCommandPath) -ChildPath ($title + '_log.txt')
$PC = $env:COMPUTERNAME
$runTime = Get-Date -Format "yyyy-MM-dd HH:mm:ss"

# defined items (reference only)
$AT = 'AudioTriggers'
$DIR = 'Dir'
$CHAROTH = 'CharOther'
$CHARUI = 'CharUI'
$CSTMAPS = 'CustomMaps'
$EQCLI = 'EQClient'
$UIF = 'UIFiles'
$UIT = 'UITemplates'
$USR = 'UserData'


################################################################################
# User configuration
$OneDrive = $env:OneDrive
$Config = @{
    'MasterBase'      = $OneDrive + '\Vault\EQ\_master-2025'
}


<#
:: ========================================================================
::
## install data and versioning subfolders
:: one install node, adjust Install/Version to reflect your setup. Anything matching will share files

    @{
        "$DIR"     = 'C:\Users\Public\Daybreak Game Company\Installed Games\EverQuest'
        "$AT"      = @{
            Install = 'AudioTriggers'
            Version = 'Live'
        }
        "$CHAROTH" = @{
            Install = ''
            Version = 'Live'
        }
        "$CHARUI"  = @{
            Install = ''
            Version = 'Live'
            Resolution = '3440x1440-Windowed'
        }
        "$CSTMAPS" = @{
            Install = 'maps'
            Version = 'Live'
        }
        "$EQCLI"   = @{
            Install = ''
            Version = 'Live'
            Resolution = '3440x1440-Windowed'
        }
        "$UIF"     = @{
            Install = 'uifiles'
            Version = 'Live'
        }
        "$UIT"     = @{
            Install = ''
            Version = 'Live'
        }
        "$USR"     = @{
            Install = 'userdata'
            Version = 'Live'
        }
    }

:: ========================================================================
#>
$Installs = @(
    @{
        "$DIR"     = 'C:\Users\Public\Daybreak Game Company\Installed Games\EverQuest'
        "$AT"      = @{
            Install = 'AudioTriggers'
            Version = 'Live'
        }
        "$CHAROTH" = @{
            Install = ''
            Version = 'Live'
        }
        "$CHARUI"  = @{
            Install = ''
            Version = 'Live'
            Resolution = '3440x1440-Windowed'
        }
        "$CSTMAPS" = @{
            Install = 'maps'
            Version = 'Live'
        }
        "$EQCLI"   = @{
            Install = ''
            Version = 'Live'
            Resolution = '3440x1440-Windowed'
        }
        "$UIF"     = @{
            Install = 'uifiles'
            Version = 'Live'
        }
        "$UIT"     = @{
            Install = ''
            Version = 'Live'
        }
        "$USR"     = @{
            Install = 'userdata'
            Version = 'Live'
        }
    }
    @{
        "$DIR"     = 'C:\Users\Public\Daybreak Game Company\Installed Games\EverQuest-Test-Min'
        "$AT"      = @{
            Install = 'AudioTriggers'
            Version = 'Test'
        }
        "$CHAROTH" = @{
            Install = ''
            Version = 'Test'
        }
        "$CHARUI"  = @{
            Install = ''
            Version = 'Test'
            Resolution = '3440x1440-Windowed'
        }
        "$CSTMAPS" = @{
            Install = 'maps'
            Version = 'Test'
        }
        "$EQCLI"   = @{
            Install = ''
            Version = 'Test'
            Resolution = 'Min-3440x1440-Windowed'
        }
        "$UIF"     = @{
            Install = 'uifiles'
            Version = 'Test'
        }
        "$UIT"     = @{
            Install = ''
            Version = 'Test'
        }
        "$USR"     = @{
            Install = 'userdata'
            Version = 'Test'
        }
    }
)

################################################################################
################################################################################

# functions
function Get-Locs {
    param (
        [string]$SourceBase,
        [string]$SourceFolders,
        [string]$DestinationBase,
        [string]$DestinationFolders
    )

    $locs = @{"Source" = ""; "Destination" = ""}

    # Ensure paths are trimmed of trailing backslashes
    $SourceBase = $SourceBase.TrimEnd('\')
    $DestinationBase = $DestinationBase.TrimEnd('\')

    if (-not [string]::IsNullOrEmpty($SourceBase) -and -not [string]::IsNullOrEmpty($DestinationBase)) {
        # source
        $locs["Source"] = if (-not [string]::IsNullOrEmpty($SourceFolders)) {
            Join-Path -Path $SourceBase -ChildPath $SourceFolders.Trim('\')
        } else {
            $SourceBase
        }

        # destination
        $locs["Destination"] = if (-not [string]::IsNullOrEmpty($DestinationFolders)) {
            Join-Path -Path $DestinationBase -ChildPath $DestinationFolders.Trim('\')
        } else {
            $DestinationBase
        }
    }

    return $locs
}


function Write-Log {
    param (
        [string]$Message,
        [string]$LogFile
    )

    $Message | Add-Content -Path $LogFile -Encoding UTF8
}
# ========================================================================


################################################################################
# functional folders
$MasterMap = @{
    "$AT"      = 'AudioTriggers'
    "$CHAROTH" = 'Char Other'
    "$CHARUI"  = 'Char UI'
    "$EQCLI"   = 'EQ Client'
    "$UIT"     = 'UI Templates'
}

# required install nodes
$RequiredInstallNodes = @{
    "$DIR"     = ''
    "$AT"      = @{Install = 'AudioTriggers'; Version = 'Noob' }
    "$CHAROTH" = @{Install = ''; Version = 'Noob' }
    "$CHARUI"  = @{Install = ''; Version = 'Noob'; Resolution = 'Noob' }
    "$CSTMAPS" = @{Install = 'maps'; SubDir = $true; Version = 'Noob' }
    "$EQCLI"   = @{Install = ''; Version = 'Noob'; Resolution = 'Noob' }
    "$UIF"     = @{Install = 'uifiles'; Version = 'Noob' }
    "$UIT"     = @{Install = ''; Version = 'Noob' }
    "$USR"     = @{Install = 'userdata'; Version = 'Noob' }
}


################################################################################
# robocopy stuff
$RoboLog = "/LOG+:$logFile"

# Sync options (not mirror) -- powershell requires this formatting otherwise fucks up robocopy
$SyncOptions = @{
    "$AT"      = @("/S", "/min:1", "/XO")
    "$CHAROTH" = @("*.ini", "/min:1", "/XO", "/XF", "*_characters*", "defaults*", "eqclient*", "eqls*", "LaunchPad*", "Uninstaller*", "VoiceChat*", "UI*")
    "$CHARUI"  = @("UI_*_*.ini", "/min:1", "/XO")
    "$CSTMAPS" = @("/S", "/min:1", "/XO")
    "$EQCLI"   = @("eqclient.ini", "/min:1", "/XO")
    "$USR"     = @("/min:1", "/XO")
}

<# Mirror options (not sync)
:: *********************
:: ***** IMPORTANT *****
:: *********************

Setup the master first to have appropriate directories for the MIRROR!!!!!
And make sure you have these for each new EQ install!!!!!

:: *********************
:: ***** IMPORTANT *****
:: *********************
#>
$MirrorOptions = @{
    "$UIF" = @("/S", "/XD", "Bigzz_THJ_UI", "/XD", "Blue", "/XD", "classic_spell_icons", "/XD", "default", "/XD", "gearcore", "/XD", "sneaky_defiance_ui", "/XD", "thj_ui", "/min:1", "/MIR", "/XO")
    #"$UIT" = @("UI*template*.ini", "/min:1", "/MIR", "/XO")
}


<# Pattern Mirror options (not sync)
:: *********************
:: ***** IMPORTANT *****
:: *********************

Setup the master first to have appropriate directories for the MIRROR!!!!!
And make sure you have these for each new EQ install!!!!!

:: *********************
:: ***** IMPORTANT *****
:: *********************
#>
$PatternMirrorOptions = @{
    "$UIT" = @{
        Pattern = "UI*template*.ini"
        Options = ("/min:1", "/XO")
    }
}


# SubDirectory folder exclusions
$SubDirExclusions = @{
    "$CSTMAPS" = @("THJ Darkmode")
}


################################################################################
# Initialize log file
$title | Set-Content -Path $logFile -Encoding UTF8
Write-Log -Message "PC: $PC" -LogFile $logFile
Write-Log -Message "Time: $runTime" -LogFile $logFile
Write-Log -Message "" -LogFile $logFile


# Logging
#Start-Transcript -Path "$logFile" -Append


try {
    # validate $Installs is filled out, use defaults
    foreach ($installNode in $Installs) {

        # check required hash
        foreach ($requiredKey in $RequiredInstallNodes.Keys) {
            $requiredNode = $RequiredInstallNodes[$requiredKey]
            if ($requiredNode -is [hashtable]) {

                # check if $Installs node is hashtable, init if not
                if (-not $installNode.ContainsKey($requiredKey) -or -not ($installNode[$requiredKey] -is [hashtable])) {
                    $installNode[$requiredKey] = @{}
                }

                $installSubNode = $installNode[$requiredKey]

                # validation
                foreach ($requiredNodeKey in $requiredNode.Keys) {
                    if (-not $installSubNode.ContainsKey($requiredNodeKey)) {
                        $installSubNode[$requiredNodeKey] = $requiredNode[$requiredNodeKey]
                    }
                }
            }
        }
    }

    if (-not [string]::IsNullOrEmpty($Config["MasterBase"])) {
        $MasterBase = $Config["MasterBase"]

        # grandmaster to master
        # if ($config.UseGrandMaster == $true) {
        #     robocopy "$($Config.GrandMasterBase)" "$($Config.MasterBase)" $optionNewOnly
        # }

        # master and install(s)
        foreach ($install in $Installs) {
            if (-not [string]::IsNullOrEmpty($install[$DIR])) {
                $InstallDir = $install[$DIR]

                # sync
                foreach ($node in $SyncOptions.Keys) {
                    $Options = $SyncOptions[$node]

                    # folder building
                    $MasterFolders = if ($MasterMap.ContainsKey($node)) {
                        $install[$node]["Version"] + '\' + $MasterMap[$node]
                    }
                    else {
                        $install[$node]["Version"] + '\' + $install[$node]["Install"]
                    }

                    # Add resolution if needed
                    if ($install[$node].Resolution) { $MasterFolders += "\$($install[$node].Resolution)" }

                    $InstallFolders = $install[$node]["Install"]

                    # get the locations
                    $Paths = Get-Locs -SourceBase $MasterBase -SourceFolders $MasterFolders -DestinationBase $InstallDir -DestinationFolders $InstallFolders
                    $MasterLoc = $Paths["Source"]
                    $InstallLoc = $Paths["Destination"]

                    # 2 way sync
                    if ($install[$node].ContainsKey("SubDir") -and $install[$node]["SubDir"]) {
                        # master to install
                        if (Test-Path -Path "$($MasterLoc)") {
                            $SubDirs = Get-ChildItem -Path "$($MasterLoc)" -Directory
                            foreach ($Directory in $SubDirs) {
                                if ($SubDirExclusions.ContainsKey($node) -and $SubDirExclusions[$node] -contains $Directory.Name) {
                                    continue  # Skip this directory and continue to the next iteration
                                }
                                $InstallLocDir = $InstallLoc + '\' + $Directory.Name
                                robocopy "$($Directory.FullName)" "$($InstallLocDir)" $Options $RoboLog
                            }
                        }


                        # install to master
                        if (Test-Path -Path "$($InstallLoc)") {
                            $SubDirs = Get-ChildItem -Path "$($InstallLoc)" -Directory
                            foreach ($Directory in $SubDirs) {
                                if ($SubDirExclusions.ContainsKey($node) -and $SubDirExclusions[$node] -contains $Directory.Name) {
                                    continue  # Skip this directory and continue to the next iteration
                                }
                                $MasterLocDir = $MasterLoc + '\' + $Directory.Name
                                robocopy "$($Directory.FullName)" "$($MasterLocDir)" $Options $RoboLog
                            }
                        }
                    }
                    else {
                        if (Test-Path -Path "$($MasterLoc)") {
                            robocopy "$MasterLoc" "$InstallLoc" $Options $RoboLog
                        }
                        if (Test-Path -Path "$($InstallLoc)") {
                            robocopy "$InstallLoc" "$MasterLoc" $Options $RoboLog
                        }
                    }
                }

                # mirror
                foreach ($mirror in $MirrorOptions.Keys) {
                    $Options = $MirrorOptions[$mirror]

                    # folder building
                    $MasterFolders = if ($MasterMap.ContainsKey($mirror)) {
                        $install[$mirror]["Version"] + '\' + $MasterMap[$mirror]
                    }
                    else {
                        $install[$mirror]["Version"] + '\' + $install[$mirror]["Install"]
                    }

                    # Add resolution if needed
                    if ($install[$mirror].Resolution) { $MasterFolders += "\$($install[$mirror].Resolution)" }

                    $InstallFolders = $install[$mirror]["Install"]

                    # get the locations
                    $Paths = Get-Locs -SourceBase $MasterBase -SourceFolders $MasterFolders -DestinationBase $InstallDir -DestinationFolders $InstallFolders
                    $MasterLoc = $Paths["Source"]
                    $InstallLoc = $Paths["Destination"]

                    # mirror
                    if (Test-Path -Path "$($MasterLoc)") {
                        robocopy "$MasterLoc" "$InstallLoc" $Options $RoboLog
                    }
                }

                # pattern mirror
                foreach ($patternMirror in $PatternMirrorOptions.Keys) {
                    $Options = $PatternMirrorOptions[$patternMirror]

                    # folder building
                    $MasterFolders = if ($MasterMap.ContainsKey($patternMirror)) {
                        $install[$patternMirror]["Version"] + '\' + $MasterMap[$patternMirror]
                    }
                    else {
                        $install[$patternMirror]["Version"] + '\' + $install[$patternMirror]["Install"]
                    }

                    # Add resolution if needed
                    if ($install[$patternMirror].Resolution) { $MasterFolders += "\$($install[$patternMirror].Resolution)" }

                    $InstallFolders = $install[$patternMirror]["Install"]

                    # get the locations
                    $Paths = Get-Locs -SourceBase $MasterBase -SourceFolders $MasterFolders -DestinationBase $InstallDir -DestinationFolders $InstallFolders
                    $MasterLoc = $Paths["Source"]
                    $InstallLoc = $Paths["Destination"]

                    # pattern mirror logic
                    if (Test-Path -Path "$($MasterLoc)") {
                        robocopy "$MasterLoc" "$InstallLoc" $Options.Pattern $Options.Options $RoboLog

                        # Cleanup install
                        # $sourceFiles = Get-ChildItem -Path "$MasterLoc" -Filter $Options.Pattern -File -Recurse
                        $destFiles = Get-ChildItem -Path "$InstallLoc" -Filter $Options.Pattern -File -Recurse

                        $destFiles | ForEach-Object {
                            $relativePath = $_.FullName.Substring($InstallLoc.Length).TrimStart('\')
                            $correspondingSource = Join-Path $MasterLoc $relativePath

                            if (-not (Test-Path $correspondingSource)) {
                                Remove-Item -Path $_.FullName -Force
                                Write-Log -Message "Removed file: $($_.FullName)" -LogFile $logFile
                            }
                        }

                        # top level matching, not recursive
                        # $destFiles | Where-Object {
                        #     $sourceFiles.Name -notcontains $_.Name
                        # } | ForEach-Object {
                        #     Remove-Item -Path $_.FullName -Force
                        #     Write-Log -Message "Removed file: $($_.FullName)" -LogFile $logFile
                        # }
                    }
                }
            }
        }
    }
} catch {
    Write-Error "$_"
} finally {
    # Cleanup actions: Stop the transcript
    # if (Test-Path "$logFile") {
    #     try {
    #         Stop-Transcript
    #         Write-Output "Transcript stopped."
    #     }
    #     catch {
    #         Write-Warning "Failed to stop transcript: $_"
    #     }
    # }
    # else {
    #     Write-Verbose "No active transcript file found at $logFile."
    # }
}
